<?php
declare(strict_types=1);

function normalize_phone(string $p): string {
    $p = preg_replace('/\D/', '', $p);
    if (str_starts_with($p, '98')) $p = substr($p, 2);
    if (str_starts_with($p, '0')) $p = substr($p, 1);
    return '0' . $p;
}

function tracking_code(int $len = 8): string {
    $chars = '23456789ABCDEFGHJKLMNPQRSTUVWXYZ';
    $out = '';
    for ($i=0; $i<$len; $i++) {
        $out .= $chars[random_int(0, strlen($chars)-1)];
    }
    return $out;
}

function weekday_from_date(string $ymd): int {
    return (int)date('w', strtotime($ymd));
}

function get_persian_date_label(string $ymd): string {
    if (!class_exists('IntlDateFormatter')) return $ymd;
    $fmt = new IntlDateFormatter(
        'fa_IR@calendar=persian',
        IntlDateFormatter::FULL,
        IntlDateFormatter::NONE,
        'Asia/Tehran',
        IntlDateFormatter::TRADITIONAL,
        'EEEE d MMMM'
    );
    return $fmt->format(new DateTime($ymd));
}

function format_datetime_jalali(string $datetime): string {
    if (!class_exists('IntlDateFormatter')) return $datetime;
    $fmt = new IntlDateFormatter(
        'fa_IR@calendar=persian',
        IntlDateFormatter::SHORT,
        IntlDateFormatter::SHORT,
        'Asia/Tehran',
        IntlDateFormatter::TRADITIONAL,
        'yyyy/MM/dd HH:mm'
    );
    return $fmt->format(new DateTime($datetime));
}

function csrf_token(): string {
    if (empty($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

function csrf_check(): void {
    $token = $_POST['csrf'] ?? '';
    if (!$token || $token !== ($_SESSION['csrf_token'] ?? '')) {
        http_response_code(403);
        die('خطای امنیتی CSRF');
    }
}

function e(string $s): string {
    return htmlspecialchars($s, ENT_QUOTES, 'UTF-8');
}

/**
 * توابع مشترک دیتابیس برای محاسبه نوبت‌ها
 */
function get_services(PDO $pdo): array {
  return $pdo->query("SELECT s.id, s.title, s.duration_minutes, s.operator_id, o.name AS operator_name
                      FROM services s JOIN operators o ON o.id=s.operator_id
                      WHERE s.is_active=1 AND o.is_active=1
                      ORDER BY s.id")->fetchAll();
}

function get_working_windows(PDO $pdo, int $operator_id, int $weekday): array {
  $st = $pdo->prepare("SELECT start_time, end_time FROM working_hours
                       WHERE operator_id=? AND weekday=? ORDER BY start_time");
  $st->execute([$operator_id, $weekday]);
  return $st->fetchAll();
}

function is_blocked(PDO $pdo, int $operator_id, DateTime $start, DateTime $end): bool {
  $st = $pdo->prepare("SELECT 1 FROM blocks
                       WHERE operator_id=?
                       AND NOT (end_at <= ? OR start_at >= ?)
                       LIMIT 1");
  $st->execute([$operator_id, $start->format('Y-m-d H:i:s'), $end->format('Y-m-d H:i:s')]);
  return (bool)$st->fetchColumn();
}

// جلوگیری از رزرو همزمان (بررسی تداخل زمانی دقیق)
function is_taken(PDO $pdo, int $operator_id, DateTime $start, DateTime $end): bool {
  $st = $pdo->prepare("SELECT 1 FROM appointments
                       WHERE operator_id=? AND status='CONFIRMED'
                       AND start_at < ? AND end_at > ?
                       LIMIT 1");
  $st->execute([$operator_id, $end->format('Y-m-d H:i:s'), $start->format('Y-m-d H:i:s')]);
  return (bool)$st->fetchColumn();
}

function available_slots(PDO $pdo, int $service_id, string $ymd): array {
  $sv = $pdo->prepare("SELECT id,title,duration_minutes,operator_id FROM services WHERE id=? AND is_active=1");
  $sv->execute([$service_id]);
  $service = $sv->fetch();
  if (!$service) return [];

  $weekday = weekday_from_date($ymd);
  $windows = get_working_windows($pdo, (int)$service['operator_id'], $weekday);
  if (!$windows) return [];

  $duration = (int)$service['duration_minutes'];
  $slots = [];
  $now = new DateTime('now', new DateTimeZone('Asia/Tehran'));

  foreach ($windows as $w) {
    $startTime = new DateTime("$ymd {$w['start_time']}", new DateTimeZone('Asia/Tehran'));
    $endTime = new DateTime("$ymd {$w['end_time']}", new DateTimeZone('Asia/Tehran'));

    $cursor = clone $startTime;
    while (true) {
      $endSlot = (clone $cursor)->modify("+{$duration} minutes");
      if ($endSlot > $endTime) break;

      // جلوگیری از رزرو ساعت‌های گذشته در امروز
      if ($cursor < $now) {
          $cursor->modify('+' . SLOT_MINUTES . ' minutes');
          continue;
      }

      // بررسی باز بودن و تداخل نداشتن این تایم
      if (!is_blocked($pdo, (int)$service['operator_id'], $cursor, $endSlot) && !is_taken($pdo, (int)$service['operator_id'], $cursor, $endSlot)) {
        $slots[] = $cursor->format('H:i');
      }
      $cursor->modify('+' . SLOT_MINUTES . ' minutes');
    }
  }
  return $slots;
}

function ymd_list(int $days): array {
  $out = [];
  $d = new DateTime('today', new DateTimeZone('Asia/Tehran'));
  for ($i=0;$i<$days;$i++){
    $out[] = [
      'value' => $d->format('Y-m-d'),
      'label' => get_persian_date_label($d->format('Y-m-d')),
      'day_name' => explode(' ', get_persian_date_label($d->format('Y-m-d')))[0] ?? '',
      'day_number' => explode(' ', get_persian_date_label($d->format('Y-m-d')))[1] ?? '',
      'month_name' => explode(' ', get_persian_date_label($d->format('Y-m-d')))[2] ?? ''
    ];
    $d->modify('+1 day');
  }
  return $out;
}

/**
 * ارسال پیامک پترن (Verify) با sms.ir
 */
function send_sms(string $mobile, int $templateId, array $parameters): bool {
    $params = [];
    foreach ($parameters as $key => $value) {
        $params[] = ['name' => $key, 'value' => (string)$value];
    }

    $data = [
        "mobile" => $mobile,
        "templateId" => $templateId,
        "parameters" => $params
    ];

    $curl = curl_init();
    curl_setopt_array($curl, [
        CURLOPT_URL => "https://api.sms.ir/v1/send/verify",
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_ENCODING => "",
        CURLOPT_MAXREDIRS => 10,
        CURLOPT_TIMEOUT => 10,
        CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
        CURLOPT_CUSTOMREQUEST => "POST",
        CURLOPT_POSTFIELDS => json_encode($data),
        CURLOPT_HTTPHEADER => [
            "Content-Type: application/json",
            "Accept: text/plain",
            "x-api-key: " . SMS_API_KEY
        ],
    ]);

    $response = curl_exec($curl);
    $err = curl_error($curl);
    curl_close($curl);

    if ($err) {
        error_log("SMS Error: " . $err);
        return false;
    }
    
    $result = json_decode($response, true);
    if (isset($result['status']) && $result['status'] == 1) {
        return true;
    }
    
    error_log("SMS API Response: " . $response);
    return false;
}