<?php
declare(strict_types=1);
require_once __DIR__ . '/db.php';
require_once __DIR__ . '/utils.php';
require_once __DIR__ . '/config.php';

if (session_status() !== PHP_SESSION_ACTIVE) session_start();
$pdo = db();

$action = $_GET['action'] ?? 'home';
$msg = $_SESSION['msg'] ?? null;
unset($_SESSION['msg']);

if ($action === 'get_slots') {
    header('Content-Type: application/json');
    $sid = (int)($_GET['service_id'] ?? 0);
    $date = (string)($_GET['ymd'] ?? '');
    echo json_encode(['slots' => available_slots($pdo, $sid, $date)]);
    exit;
}

$services = get_services($pdo);
$dates = ymd_list(BOOK_AHEAD_DAYS);

// رزرو نهایی
if ($action === 'book' && $_SERVER['REQUEST_METHOD'] === 'POST') {
  csrf_check();
  $service_id = (int)($_POST['service_id'] ?? 0);
  $ymd = trim((string)($_POST['ymd'] ?? ''));
  $time = trim((string)($_POST['time'] ?? ''));
  $name = trim((string)($_POST['full_name'] ?? ''));
  $phone = normalize_phone((string)($_POST['phone'] ?? ''));

  if (!$service_id || !$ymd || !$time || !$name || strlen($phone) < 10) {
    $_SESSION['msg'] = 'اطلاعات ناقص است.';
    header('Location: index.php');
    exit;
  }

  $st = $pdo->prepare("SELECT id,title,duration_minutes,operator_id FROM services WHERE id=? AND is_active=1");
  $st->execute([$service_id]);
  $service = $st->fetch();
  if (!$service) { $_SESSION['msg']='سرویس نامعتبر است.'; header('Location: index.php'); exit; }

  $today = new DateTime('today', new DateTimeZone('Asia/Tehran'));
  $max = (clone $today)->modify('+' . (BOOK_AHEAD_DAYS-1) . ' days');
  $d = new DateTime($ymd, new DateTimeZone('Asia/Tehran'));
  if ($d < $today || $d > $max) { $_SESSION['msg']='تاریخ خارج از بازه مجاز است.'; header('Location: index.php'); exit; }

  $slots = available_slots($pdo, $service_id, $ymd);
  if (!in_array($time, $slots, true)) { $_SESSION['msg']='این تایم پر شده یا دیگر در دسترس نیست.'; header('Location: index.php'); exit; }

  $start = new DateTime("$ymd $time:00", new DateTimeZone('Asia/Tehran'));
  $end = (clone $start)->modify('+' . (int)$service['duration_minutes'] . ' minutes');

  $pdo->beginTransaction();
  try {
    $st = $pdo->prepare("SELECT id FROM customers WHERE phone=? LIMIT 1");
    $st->execute([$phone]);
    $existing_customer = $st->fetchColumn();

    if ($existing_customer) {
        $customer_id = (int)$existing_customer;
    } else {
        $st = $pdo->prepare("INSERT INTO customers(full_name, phone) VALUES(?, ?)");
        $st->execute([$name, $phone]);
        $customer_id = (int)$pdo->lastInsertId();
    }

    $code = tracking_code(10);
    $st = $pdo->prepare("INSERT INTO appointments(customer_id, service_id, operator_id, start_at, end_at, tracking_code, status)
                         VALUES(?,?,?,?,?,?, 'CONFIRMED')");
    $st->execute([
      $customer_id,
      (int)$service['id'],
      (int)$service['operator_id'],
      $start->format('Y-m-d H:i:s'),
      $end->format('Y-m-d H:i:s'),
      $code
    ]);

    $pdo->commit();
    
    // تبدیل تاریخ به شمسی خوانا (مثال: 7 اسفند)
    $persian_date_full = get_persian_date_label($ymd);
    $date_parts = explode(' ', $persian_date_full);
    $short_jalali_date = (isset($date_parts[1]) && isset($date_parts[2])) ? $date_parts[1] . ' ' . $date_parts[2] : $ymd;
    
    $sms_template_id = 100000; // شناسه پترن اس ام اس خود را اینجا قرار دهید
    send_sms($phone, $sms_template_id, [
        'name' => $name,
        'date' => $short_jalali_date,
        'time' => $time,
        'code' => $code
    ]);

    $_SESSION['msg'] = "نوبت شما ثبت شد ✅ کد رهگیری: $code";
  } catch (Throwable $e) {
    $pdo->rollBack();
    $_SESSION['msg'] = "خطای دیتابیس: " . $e->getMessage();
  }

  header('Location: index.php');
  exit;
}

// پیگیری/لغو
if ($action === 'lookup' && $_SERVER['REQUEST_METHOD'] === 'POST') {
  csrf_check();
  $phone = normalize_phone((string)($_POST['phone'] ?? ''));
  $code = trim((string)($_POST['tracking_code'] ?? ''));
  if (strlen($phone) < 10 || strlen($code) < 6) {
    $_SESSION['msg'] = 'شماره یا کد رهگیری نامعتبر است.';
    header('Location: index.php#lookup');
    exit;
  }

  $_SESSION['lookup_phone'] = $phone;
  $_SESSION['lookup_code'] = $code;
  header('Location: index.php#lookup');
  exit;
}

if ($action === 'cancel' && $_SERVER['REQUEST_METHOD'] === 'POST') {
  csrf_check();
  $appt_id = (int)($_POST['appointment_id'] ?? 0);
  $phone = normalize_phone((string)($_POST['phone'] ?? ''));
  $code = trim((string)($_POST['tracking_code'] ?? ''));

  $st = $pdo->prepare("SELECT a.id FROM appointments a
                       JOIN customers c ON c.id=a.customer_id
                       WHERE a.id=? AND a.tracking_code=? AND c.phone=? AND a.status='CONFIRMED' LIMIT 1");
  $st->execute([$appt_id, $code, $phone]);
  $ok = $st->fetchColumn();

  if ($ok) {
    $up = $pdo->prepare("UPDATE appointments SET status='CANCELED' WHERE id=?");
    $up->execute([$appt_id]);
    $_SESSION['msg'] = 'نوبت لغو شد ✅';
  } else {
    $_SESSION['msg'] = 'اطلاعات برای لغو معتبر نیست.';
  }
  header('Location: index.php#lookup');
  exit;
}

$myAppointments = [];
$lookup_phone = $_SESSION['lookup_phone'] ?? '';
$lookup_code = $_SESSION['lookup_code'] ?? '';
if ($lookup_phone && $lookup_code) {
  $st = $pdo->prepare("SELECT a.id, a.start_at, a.end_at, a.status, a.tracking_code, s.title
                       FROM appointments a
                       JOIN customers c ON c.id=a.customer_id
                       JOIN services s ON s.id=a.service_id
                       WHERE c.phone=? AND a.tracking_code=?
                       ORDER BY a.start_at DESC");
  $st->execute([$lookup_phone, $lookup_code]);
  $myAppointments = $st->fetchAll();
}

$selected_service = (int)($_GET['service_id'] ?? ($services[0]['id'] ?? 0));
$selected_date = (string)($_GET['ymd'] ?? ($dates[0]['value'] ?? date('Y-m-d')));
$slots = ($selected_service && $selected_date) ? available_slots($pdo, $selected_service, $selected_date) : [];
?>
<!doctype html>
<html lang="fa" dir="rtl">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1, viewport-fit=cover, maximum-scale=1, user-scalable=no">
  <title><?= e(APP_NAME) ?> | رزرو نوبت</title>
  <link href="https://cdn.jsdelivr.net/npm/remixicon@3.5.0/fonts/remixicon.css" rel="stylesheet">
  <link rel="stylesheet" href="style.css">
  <link rel="stylesheet" href="assets/booking-modern.css?v=6">
  <script src="assets/booking-modern.js?v=6" defer></script>
</head>
<body class="app-shell">

<div class="app-container">
  <header class="app-header">
    <div class="brand">
        <h1><?= e(APP_NAME) ?></h1>
        <p>رزرو آنلاین سریع و آسان</p>
    </div>
    <button class="theme-toggle" id="theme-toggle" aria-label="تغییر تم">
        <i class="ri-moon-fill dark-icon"></i>
        <i class="ri-sun-fill light-icon"></i>
    </button>
  </header>

  <?php if ($msg): ?>
    <div class="notification animate-pop">
        <i class="ri-notification-3-line"></i>
        <span><?= e($msg) ?></span>
    </div>
  <?php endif; ?>

  <nav class="nav-tabs">
    <button type="button" class="nav-item active" data-target="book-tab">
      <i class="ri-calendar-add-line"></i> ثبت نوبت
    </button>
    <button type="button" class="nav-item" data-target="lookup-tab">
      <i class="ri-search-eye-line"></i> پیگیری
    </button>
  </nav>

  <main id="book-tab" class="tab-content active">
    
    <section class="section">
      <h3 class="section-title"><i class="ri-service-line"></i> انتخاب خدمت</h3>
      <div class="service-scroller">
        <?php foreach($services as $s): ?>
          <?php $isActive = $selected_service === (int)$s['id']; ?>
          <a href="#" data-id="<?= (int)$s['id'] ?>" class="service-card <?= $isActive ? 'active' : '' ?>">
            <div class="service-icon"><i class="ri-scissors-cut-line"></i></div>
            <div class="service-info">
              <span class="service-name"><?= e($s['title']) ?></span>
              <span class="service-meta"><?= (int)$s['duration_minutes'] ?> دقیقه • <?= e($s['operator_name']) ?></span>
            </div>
            <div class="check-icon" style="display: <?= $isActive ? 'block' : 'none' ?>"><i class="ri-check-line"></i></div>
          </a>
        <?php endforeach; ?>
      </div>
    </section>

    <section class="section">
      <h3 class="section-title"><i class="ri-calendar-event-line"></i> انتخاب تاریخ</h3>
      <div class="date-scroller">
        <?php foreach($dates as $d): ?>
          <?php $isActive = $selected_date === $d['value']; ?>
          <a href="#" data-ymd="<?= e($d['value']) ?>" class="date-pill <?= $isActive ? 'active' : '' ?>">
            <span class="day-name"><?= e($d['day_name']) ?></span>
            <span class="day-number"><?= e($d['day_number']) ?></span>
            <span class="month-name"><?= e($d['month_name']) ?></span>
          </a>
        <?php endforeach; ?>
      </div>
    </section>

    <section class="section">
      <h3 class="section-title"><i class="ri-time-line"></i> انتخاب ساعت و ثبت</h3>
      
      <div class="empty-state" id="empty-slots-msg" style="display: <?= empty($slots) ? 'block' : 'none' ?>">
        <i class="ri-calendar-close-line"></i>
        <p>متاسفانه برای تاریخ انتخاب شده، نوبت خالی وجود ندارد.</p>
      </div>

      <form method="post" action="index.php?action=book" class="booking-form" id="booking-form" style="display: <?= empty($slots) ? 'none' : 'block' ?>">
        <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
        <input type="hidden" name="service_id" id="form_service_id" value="<?= (int)$selected_service ?>">
        <input type="hidden" name="ymd" id="form_ymd" value="<?= e($selected_date) ?>">

        <div class="time-grid" id="time-grid-container">
            <?php foreach($slots as $t): ?>
            <label class="time-radio">
                <input type="radio" name="time" value="<?= e($t) ?>" required>
                <span class="time-badge"><?= e($t) ?></span>
            </label>
            <?php endforeach; ?>
        </div>

        <div class="form-group-wrapper">
            <div class="input-group">
                <i class="ri-user-smile-line input-icon"></i>
                <input class="input-field" name="full_name" placeholder="نام و نام خانوادگی" required>
            </div>
            <div class="input-group">
                <i class="ri-smartphone-line input-icon"></i>
                <input class="input-field" type="tel" name="phone" placeholder="شماره موبایل (۰۹...)" pattern="[0-9]*" inputmode="numeric" required>
            </div>
        </div>

        <button class="btn-primary" type="submit">
            <span>تایید و رزرو نوبت</span>
            <i class="ri-arrow-left-s-line"></i>
        </button>
      </form>
    </section>
  </main>

  <main id="lookup-tab" class="tab-content">
    <div class="lookup-card">
        <h2 class="lookup-title">پیگیری وضعیت نوبت</h2>
        <form method="post" action="index.php?action=lookup">
            <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
            <div class="input-group">
                <i class="ri-smartphone-line input-icon"></i>
                <input class="input-field" name="phone" value="<?= e($lookup_phone) ?>" placeholder="شماره موبایل" type="tel" required>
            </div>
            <div class="input-group" style="margin-top:10px">
                <i class="ri-barcode-line input-icon"></i>
                <input class="input-field" name="tracking_code" value="<?= e($lookup_code) ?>" placeholder="کد رهگیری" required>
            </div>
            <button class="btn-secondary" type="submit">جستجو</button>
        </form>
    </div>

    <?php if ($myAppointments): ?>
        <div class="appointments-list">
        <?php foreach($myAppointments as $a): ?>
            <div class="appt-card">
                <div class="appt-header">
                    <span class="appt-service"><?= e($a['title']) ?></span>
                    <span class="appt-status status-<?= strtolower($a['status']) ?>"><?= $a['status'] === 'CONFIRMED' ? 'فعال' : 'لغو شده' ?></span>
                </div>
                <div class="appt-details">
                    <div><i class="ri-calendar-line"></i> <?= e(format_datetime_jalali($a['start_at'])) ?></div>
                    <div><i class="ri-barcode-box-line"></i> <?= e($a['tracking_code']) ?></div>
                </div>
                <?php if ($a['status'] === 'CONFIRMED'): ?>
                    <form method="post" action="index.php?action=cancel">
                        <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
                        <input type="hidden" name="appointment_id" value="<?= (int)$a['id'] ?>">
                        <input type="hidden" name="phone" value="<?= e($lookup_phone) ?>">
                        <input type="hidden" name="tracking_code" value="<?= e($lookup_code) ?>">
                        <button class="btn-text-danger" type="submit">لغو این نوبت</button>
                    </form>
                <?php endif; ?>
            </div>
        <?php endforeach; ?>
        </div>
    <?php endif; ?>
  </main>

  <footer class="app-footer">
    <a href="admin/login.php">ورود مدیریت</a>
  </footer>
</div>

</body>
</html>